 /**
 *
 * \file        HPRFGW_RFResultCode.c
 *
 * \brief       This file contains the RF device retry timer and processing functions		
 *
 * \detail		
 *
 * \author      Hazrat Shah	
 *
 * \date        10/10/2006
 * \warning	
 * \note
 * \review
 */

#include "hprfgw_config.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfApp.h"
#include "smac_app.h"
#include "hprfgw_rfRetry.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfDiagnostics.h"

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS	
////////////////////////////////////////////////////////////////////////////////
#define	RF_RETRY_TASK_RATE			100			//	100msec
#define RF_RETRY_TASK_STACK_SIZE	(8*1024)
#ifndef RF_TASKS_BASE_PRIORITY
 #if defined (OS_NUCLEUS)
  #pragma info "!!!!Pass in the RF_TASKS_BASE_PRIORITY using the make file!!!!"
 #endif
#endif
#define RF_RETRY_TASK_PRIORITY		RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY, 2)

// Enable inactivity PHY reset patch for slave device
#define RF_PHY_RX_LOCKUP_INACTIVITTY_PATCH	1

////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA		
////////////////////////////////////////////////////////////////////////////////
typedef struct	{
#if defined (OS_NUCLEUS)
	NU_TASK 	task;
	UINT8   	stack[RF_RETRY_TASK_STACK_SIZE];
#elif defined (OS_WINCE)
	HANDLE	    task;
#endif
	DM_RESOURCE_HANDLE task;
}	TYPE_RFRETRY_TASK;
TYPE_RFRETRY_TASK RFRetryTask;

UINT32 Rfdiags_NumRxInactivityResets;
UINT32 Rfdiags_getNumRxInactivityResets(void) { return Rfdiags_NumRxInactivityResets; }
void   Rfdiags_clearNumRxInactivityResets(void) { Rfdiags_NumRxInactivityResets = 0; }

////////////////////////////////////////////////////////////////////////////////
//	
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The retry delay task   
 *
 * \detail		This function checks the retry delay task of all devices whose link state is either active or TempConn 
 *				If the device retry delay time expires the retry timed out flag and Retry Timeout Event gets set. 
 *				The rf receiver task waits on the retry timeout event 
 *
 * \date        09/13/2006
 *
 * \param	    argc, argv				
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
#if defined (OS_NUCLEUS)
void	RFRetry_Task (UINT32 argc, void * argv)	{
#elif defined (OS_WINCE)
DWORD	RFRetry_Task (void* pContext)	{
#endif
static void RFRetry_Task(UINT32 param)
{
	UINT16	i;
	UINT8	AnyDeviceRetryTimeoutOccuredFlag;	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;
	UINT32 u32RFCounter = 0;
	
	RFIntHostApp.pF_StartupWait(HPRF_STARTUP_RETRY_READY);
	
	//	Wait for RF Stack to be intialized
	while (!RFReceive_IsRFStackInitialized())	{	HwDelayMsec(100);	}

#if defined (OS_WINCE)
	RFIntHostApp.pF_ErrorPrintf("\r\nRFRetry_Task: Running!");
#endif

	while (1)	{
		
		AnyDeviceRetryTimeoutOccuredFlag = FALSE;
		pdst = g_pRFDeviceDescriptiveTable;

		for (i=0; i<g_MaxDescriptiveTableEntries; i++, pdst++)	{
			if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY)  || (i == RFDesc_GetASPDescriptiveTableIndex())
																 || (i == RFDesc_GetMLMEDescriptiveTableIndex()))	{
				if (pdst->RetryTimer)	{
					if (--pdst->RetryTimer == 0)	{
						pdst->RetryTimedOut = TRUE;
						AnyDeviceRetryTimeoutOccuredFlag = TRUE;	
					}	
				}
			}		
		}

		if (AnyDeviceRetryTimeoutOccuredFlag)	{
			RFReceive_SetRetryTimeoutEvent();	
		} 	
		
#ifndef	RF_NETWORK_MASTER_DEVICE
	#ifdef RF_PHY_RX_LOCKUP_INACTIVITTY_PATCH
		//	note: Once in a while the RF PHY locks up in the slave device (TPS6x panel). 
		//	This patch issues one PHY reset request for the SMAC if there were 
		//	no packets received by SMAC for the fast 1.5second. 
		{

#if defined (OS_NUCLEUS)
			#define		RF_PHY_LOCKUP_INACTIVITTY_CHK		5		//	500msec
			#define		RF_PHY_LOCKUP_INACTIVITTY_TIMEOUT	150		//	1.5 second
#elif defined (OS_WINCE)
			#define		RF_PHY_LOCKUP_INACTIVITTY_CHK		50		//	500msec
			#define		RF_PHY_LOCKUP_INACTIVITTY_TIMEOUT	1500	//	1.5 second
#endif
			
#define		RF_PHY_LOCKUP_INACTIVITTY_CHK		500		//	500msec
#define		RF_PHY_LOCKUP_INACTIVITTY_TIMEOUT	1500	//	1.5 second
	 
			if (u32RFCounter >= RF_PHY_LOCKUP_INACTIVITTY_CHK)
			{
				u32RFCounter = 0;
				if (RFReceive_IsControlSystemConnected() && (!RFAPP_AcquireStatus()))
				{
					if (Smac_rxPktSinceReset() == TRUE)
					{
						if (OS_RetrieveClock() - Smac_getLastPktRxTime() > RF_PHY_LOCKUP_INACTIVITTY_TIMEOUT)
						{
							RFTransmit_BuildResetMACMessageAndPostToPipe(FALSE);
							Rfdiags_NumRxInactivityResets++;
							//RFIntHostApp.pF_ConsolePrintf ("RF Stack reset\r\n");
						}
					}
				}
			}
			u32RFCounter++;
		}
	#endif
#endif

		HwDelayMsec (RF_RETRY_TASK_RATE);
	}
}

////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The retry delay timer processing   
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    None		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFRetry_TimeoutProcessing (void)	{
	UINT16	i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	BOOL	allpktsexpired;

	for (i=0; i<g_MaxDescriptiveTableEntries; i++, pdst++)	{
		if (pdst->RetryTimedOut)	{
			pdst->RetryTimedOut = FALSE;
			if (pdst->txpacketretrycntr >= pdst->maxtxpacketretries)	{	

				if	((i < g_MaxSupportedRFDevices) || (i == RFDesc_GetNetworkServicesDescriptiveTableIndex()))	{
#ifdef	RF_DBG_LOG
		RFComLog_LogEvent (RF_DBG_MCP_RETRY_PACKET, i, 0, 0, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.payld.data[1], pdst->COMLog.txpacketretrycount, OS_RetrieveClock());
#endif
					
					//	Make MCP packet call-back with the result code   
					RFTransmit_MCPPacketResultCodeCallBack((UINT8)i);
					RFDesc_LinkStateTransition((UINT8)i, RF_DEVICE_LINK_STATE_FAIL);	
					RFTransmit_TriggerRFTransmitTaskEvent((UINT8)i);

	#ifdef		RF_NETWORK_MASTER_DEVICE
					//	
	#else					
					if (i == DEVICE_DESCRIPTIVE_TABLE_INDEX)	{
						if (!RFAPP_AcquireStatus())	{
							//	(05/25/07 - HS) - set state to attempting to connect to gateway
							RFDesc_LinkStateTransition ((UINT8)i, RF_DEVICE_LINK_STATE_CONNECTING);
						}
						RFAPP_ConnectionFailedwithGateway();
					}
	#endif								
				}	else	{
					if (i == RFDesc_GetASPDescriptiveTableIndex())	{
				#ifdef	RF_DBG_LOG
						RFComLog_LogEvent (RF_DBG_ASP_RETRY_PACKET, i, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.type, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.subtype, 0, pdst->COMLog.txpacketretrycount, OS_RetrieveClock());
				#endif				
					}	else	{

				#ifdef	RF_DBG_LOG
						RFComLog_LogEvent (RF_DBG_MLME_RETRY_PACKET, i, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.type, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.subtype, 0, pdst->COMLog.txpacketretrycount, OS_RetrieveClock());
				#endif				
					}
					//	inform the transmit task (ASP or MLME packet retry failure), if the retry is enabled for ASP or MLME packets
					pdst->resultcodeavailable = TRUE;				
					RFTransmit_TriggerRFTransmitTaskEvent((UINT8)i);						
				}
			}	else	{
				//	check for expired packets, make call back only for expired packets 
				RFTransmit_MCPPacketTimeToLiveExpirationChk ((UINT8)i, &allpktsexpired);					
				//	inform the transmit task to repeat packet transmission 
				pdst->resultcodeavailable = TRUE;				
				RFTransmit_TriggerRFTransmitTaskEvent((UINT8)i);					
			}
		}
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reloads the rf device retry timer	counter	 	   
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    index 		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFRetry_ReTriggerTimer (UINT8 index)	{	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	
	pdst->RetryTimedOut = FALSE;
	pdst->RetryTimer    = RFReceive_GetLocalDeviceRetryDelayAtRetryCount((UINT8)pdst->txpacketretrycntr); 
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The retry init function  
 *
 * \detail		This function creates the rf retry task	 
 *
 * \date        10/10/2006
 *
 * \param	    None		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFRetry_Init (void)	{
#if defined (OS_NUCLEUS)
	//	Create rf retry task	 
    if(NU_Create_Task(&RFRetryTask.task, "TRFRETRY", RFRetry_Task, 0, NU_NULL,
        			   RFRetryTask.stack, sizeof(RFRetryTask.stack), RF_RETRY_TASK_PRIORITY, 0, NU_PREEMPT, NU_START ) != NU_SUCCESS)	{
        RFIntHostApp.pF_ErrorHandler("TRFRTRY: Could not create task!");
    }
#elif defined (OS_WINCE)
	RFRetryTask.task = CreateThread(NULL, 0, RFRetry_Task, pHPRDevice, 0, NULL);
	if (NULL == RFRetryTask.task)
	{
		RFIntHostApp.pF_ErrorPrintf("TRFRTRY: Could not create task!");
	}
	// Set thread priority
	CeSetThreadPriority(RFRetryTask.task, RF_RETRY_TASK_PRIORITY);

#endif
	RFRetryTask.task = OsCreateNamedAdvTaskPri(RFRetry_Task, 0, 0, RF_RETRY_TASK_PRIORITY, "TRFRETRY");
	if (!RFRetryTask.task)
	{
		RFIntHostApp.pF_ErrorPrintf("TRFRETRY: Fail create task!");
	}
}

